<?php

/**
 * @title Content Image
 * @desc add preview image for protected content (file, video and page) and then display this image in Active Resources widget
 */

class Am_Plugin_ContentImage extends Am_Plugin
{
    const PLUGIN_STATUS = self::STATUS_PRODUCTION;
    const PLUGIN_COMM = self::COMM_COMMERCIAL;
    const PLUGIN_REVISION = '@@VERSION@@';

    const UPLOAD_PREFIX = 'content-image';

    protected $_configPrefix = 'misc.';

    static function getDbXml()
    {
        return <<<CUT
<schema version="4.0.0">
    <table name="file">
        <field name="img_upload_id" type="int"/>
        <field name="img_path" type="varchar" len="255"/>
        <field name="img_width" type="int" />
        <field name="img_height" type="int" />
    </table>
    <table name="page">
        <field name="img_upload_id" type="int"/>
        <field name="img_path" type="varchar" len="255"/>
        <field name="img_width" type="int" />
        <field name="img_height" type="int" />
    </table>
    <table name="video">
        <field name="img_upload_id" type="int"/>
        <field name="img_path" type="varchar" len="255"/>
        <field name="img_width" type="int" />
        <field name="img_height" type="int" />
    </table>
    <table name="folder">
        <field name="img_upload_id" type="int"/>
        <field name="img_path" type="varchar" len="255"/>
        <field name="img_width" type="int" />
        <field name="img_height" type="int" />
    </table>
    <table name="link">
        <field name="img_upload_id" type="int"/>
        <field name="img_path" type="varchar" len="255"/>
        <field name="img_width" type="int" />
        <field name="img_height" type="int" />
    </table>
</schema>
CUT;
    }

    function _initSetupForm(Am_Form_Setup $form)
    {
        $form->setTitle('Content Images');

        $form->addProlog(<<<CUT
<style type="text/css">
<!--
    .am-row:hover .color-pick {
        opacity: 1;
    }
    .color-pick {
        opacity: 0;
        display: inline-block;
        vertical-align: middle;
        cursor: pointer;
        width: 1em;
        height: 1em;
        border-radius: 50%;
        transition: transform .3s, opacity .3s;
    }
    .color-pick:hover {
        transform: scale(1.8);
    }
-->
</style>
CUT
        );

        $g = $form->addGroup()
            ->setLabel("Image Size\nwidth &times; height");

        $g->addInteger('width', ['placeholder' => 70]);
        $g->addHtml()->setHtml(' × ');
        $g->addInteger('height', ['placeholder' => 70]);

        $form->addSelect("resize_type")
            ->loadOptions([
                Am_Image::RESIZE_CROP => 'Crop',
                Am_Image::RESIZE_GIZMO => 'Gizmo',
                Am_Image::RESIZE_FITWIDTH => 'Fit Width',
                Am_Image::RESIZE_FITHEIGHT => 'Fit Height',
                Am_Image::RESIZE_FIT => 'Fit Auto',
            ])
            ->setLabel(___('Resize Type'));

        $form->addAdvRadio('bg')
            ->setLabel(___('Background'))
            ->loadOptions([
                'color' => ___('Color'),
                'transparent' => ___('Transparent')
            ]);

        $form->setDefault('bg', 'color');
        $form->setDefault('resize_type', Am_Image::RESIZE_CROP);

        $this->addElementColor($form, 'fill_color');

        $form->addUpload('default_upload_id', null, ['prefix' => self::UPLOAD_PREFIX])
            ->setLabel('Default Image');

        $form->addSaveCallbacks([$this, 'resizeDefaultImage'], null);

        $form->addScript()
            ->setScript(<<<CUT
jQuery(function(){
    jQuery('[name$=bg]').change(function(){
        jQuery('[name$=fill_color]').closest('.row, .am-row').toggle(jQuery('[name$=bg]:checked').val() == 'color');
    }).change();
});
jQuery(document).on('click', '.color-pick', function(){
    $(this).closest('.am-row').find('input').val($(this).data('color')).change();
});
jQuery(function(){
    function hexToRgb(hex) {
       var result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex);
       return result ? {
           r: parseInt(result[1], 16),
           g: parseInt(result[2], 16),
           b: parseInt(result[3], 16)
       } : null;
    }

    $('.color-input').change(function(){
        var tColor = 'inherit';

        if ((c = hexToRgb($(this).val())) &&
            (1 - (0.299 * c.r + 0.587 * c.g + 0.114 * c.b) / 255 > 0.5)) {
            tColor = '#fff';
        }
        $(this).css({background: $(this).val(), color: tColor, border: 'none'});
    }).change();
});
CUT
            );
    }

    protected function addElementColor($form, $name)
    {
        $gr = $form->addGroup()
            ->setLabel(___('Background Color'));
        $gr->setSeparator(' ');

        $attr = ['size' => 7, 'placeholder' => '#cccccc', 'class' => 'color-input'];

        $gr->addText($name, $attr);

        foreach ([
                     '#f1f5f9', '#dee7ec', '#cccccc', '#ffebcd', '#ff8a80',
                     '#ea80fc', '#d1c4e9', '#e3f2fd', '#bbdefb', '#0079d1', '#b2dfdb',
                     '#e6ee9c', '#c8e6c9', '#4caf50', '#bcaaa4', '#212121', '#263238'
                 ] as $color) {
            $gr->addHtml()
                ->setHtml("<div class='color-pick' style='background:{$color}' data-color='$color'></div>");
        }
    }

    protected function color($c)
    {
        if (preg_match('/^#([0-9a-fA-F]{6})$/', trim($c), $regs))
        {
            return hexdec($regs[1]);
        } else {
            return Am_Image::FILL_COLOR;
        }
    }

    function resizeDefaultImage(Am_Config $before, Am_Config $after)
    {
        $utoken = "{$this->_configPrefix}{$this->getId()}.default_upload_id";
        $ptoken = "{$this->_configPrefix}{$this->getId()}.default_path";
        $wtoken = "{$this->_configPrefix}{$this->getId()}.width";
        $htoken = "{$this->_configPrefix}{$this->getId()}.height";
        $ftoken = "{$this->_configPrefix}{$this->getId()}.fill_color";
        $btoken = "{$this->_configPrefix}{$this->getId()}.bg";
        $rtoken = "{$this->_configPrefix}{$this->getId()}.resize_type";

        $iwtoken = "{$this->_configPrefix}{$this->getId()}.img_width";
        $ihtoken = "{$this->_configPrefix}{$this->getId()}.img_height";

        if (!$after->get($utoken)) {
            $after->set($ptoken, null);
            $after->set($iwtoken, null);
            $after->set($ihtoken, null);
            return;
        }

        if (!$after->get($ptoken)
            || $before->get($utoken) != $after->get($utoken)
            || $before->get($htoken, 70) != $after->get($htoken, 70)
            || $before->get($wtoken, 70) != $after->get($wtoken, 70)
            || $before->get($ftoken) != $after->get($ftoken)
            || $before->get($btoken) != $after->get($btoken)
            || $before->get($rtoken, Am_Image::RESIZE_CROP) != $after->get($rtoken, Am_Image::RESIZE_CROP)
            ) {
            $after->set($ptoken, null);
            if ($after->get($utoken)) {
                $after->set($ptoken, $fn = $this->resize(
                    $after->get($utoken),
                    $after->get($htoken, 70),
                    $after->get($wtoken, 70),
                    $after->get($rtoken),
                    $after->get($btoken) == 'color' ? $this->color($after->get($ftoken)) : Am_Image::FILL_TRANSPARENT
                ));
                $img = new Am_Image($this->getDi()->public_dir . '/' . $fn);
                $after->set($iwtoken, $img->width());
                $after->set($ihtoken, $img->height());
            }
        }
    }

    function resize($upload_id, $w, $h, $resize_type, $fill)
    {
        $upload = $this->getDi()->uploadTable->load($upload_id);
        $name = str_replace('.' . self::UPLOAD_PREFIX . '.', '', $upload->path);
        $image = new Am_Image($upload->getFullPath(), $upload->getType());

        $ext = $this->getConfig('bg', 'color') == 'color' ? '.jpeg' : '.png';
        $mime = $this->getConfig('bg', 'color') == 'color' ? 'image/jpeg' : 'image/png';

        $newName = 'content/' . $w . '_' . $h . '/' . $name . $ext;
        $newFilename = $this->getDi()->public_dir . '/' . $newName;

        if (!file_exists($newFilename))
        {
            if (!is_dir(dirname($newFilename)))
            {
                mkdir(dirname($newFilename), 0777, true);
            }

            $image->resize(
                $w,
                $h,
                $resize_type,
                $fill
            )->save($newFilename, $mime);
        }
        return $newName;
    }

    function onGetUploadPrefixList(Am_Event $e)
    {
        $e->addReturn([
            Am_Upload_Acl::IDENTITY_TYPE_ADMIN => Am_Upload_Acl::ACCESS_ALL,
            Am_Upload_Acl::IDENTITY_TYPE_USER => Am_Upload_Acl::ACCESS_READ,
            Am_Upload_Acl::IDENTITY_TYPE_ANONYMOUS => Am_Upload_Acl::ACCESS_READ
        ], self::UPLOAD_PREFIX);
    }

    function onGridContentFilesInitGrid(Am_Event_Grid $e)
    {
        $this->_initGrid($e);
    }

    function onGridContentFilesInitForm(Am_Event_Grid $e)
    {
        $this->_initForm($e);
    }

    public function onGridContentFilesValuesFromForm(Am_Event_Grid $e)
    {
        $this->_valuesFromForm($e);
    }

    function onGridContentPagesInitGrid(Am_Event_Grid $e)
    {
        $this->_initGrid($e);
    }

    function onGridContentPagesInitForm(Am_Event_Grid $e)
    {
        $this->_initForm($e);
    }

    public function onGridContentPagesValuesFromForm(Am_Event_Grid $e)
    {
        $this->_valuesFromForm($e);
    }

    function onGridContentVideoInitGrid(Am_Event_Grid $e)
    {
        $this->_initGrid($e);
    }

    function onGridContentVideoInitForm(Am_Event_Grid $e)
    {
        $this->_initForm($e);
    }

    public function onGridContentVideoValuesFromForm(Am_Event_Grid $e)
    {
        $this->_valuesFromForm($e);
    }

    function onGridContentFoldersInitGrid(Am_Event_Grid $e)
    {
        $this->_initGrid($e);
    }

    function onGridContentFoldersInitForm(Am_Event_Grid $e)
    {
        $this->_initForm($e);
    }

    public function onGridContentFoldersValuesFromForm(Am_Event_Grid $e)
    {
        $this->_valuesFromForm($e);
    }

    function onGridContentLinksInitGrid(Am_Event_Grid $e)
    {
        $this->_initGrid($e);
    }

    function onGridContentLinksInitForm(Am_Event_Grid $e)
    {
        $this->_initForm($e);
    }

    public function onGridContentLinksValuesFromForm(Am_Event_Grid $e)
    {
        $this->_valuesFromForm($e);
    }

    function _initForm(Am_Event_Grid $e)
    {
        $form = $e->getGrid()->getForm();
        $form->addUpload('img_upload_id', null, ['prefix' => self::UPLOAD_PREFIX])
            ->setLabel(___('Image'));
    }

    function _valuesFromForm(Am_Event_Grid $e)
    {
        $v = $e->getArg(0);
        if (!$v['img_upload_id']) {
            $v['img_upload_id'] = null;
            $v['img_path'] = null;
            $v['img_width'] = null;
            $v['img_height'] = null;
        }
        $e->setArg(0, $v);
    }

    function _initGrid(Am_Event_Grid $e)
    {
        $e->getGrid()->addCallback(Am_Grid_Editable::CB_BEFORE_SAVE, function(& $v, $r, $g) {

            if (!$r->img_path || $r->img_upload_id != $v['img_upload_id']) {
                $r->img_path = null;
                if ($v['img_upload_id']) {
                    $r->img_path = $this->resize(
                        $v['img_upload_id'],
                        $this->getConfig('width', 70),
                        $this->getConfig('height', 70),
                        $this->getConfig('resize_type', Am_Image::RESIZE_CROP),
                        $this->getConfig('bg') == 'color' ? $this->color($this->getConfig('fill_color')) : Am_Image::FILL_TRANSPARENT
                    );
                    $img = new Am_Image($this->getDi()->public_dir . '/' . $r->img_path);
                    $r->img_width = $img->width();
                    $r->img_height = $img->height();
                }
            }

        });

        $e->getGrid()->addFieldBefore('title', 'img_path', ___('Image'), false, null, null, '1%')
            ->setRenderFunction(function($r, $fn, $g, $fo) {
                $path = $r->img_path ?: $this->getConfig('default_path');
                return $g->renderTd(
                    $path ?
                        sprintf('<img width="70" src="%s" />', $g->getDi()->url("data/public/{$path}")) :
                        '',
                    false);
            });
    }

    function onInitFinished(Am_Event $e)
    {
        if ($this->getDi()->auth->getUser()) {
            $resources = $this->getDi()->resourceAccessTable
                ->getAllowedResources(
                    $this->getDi()->user,
                    [ResourceAccess::FILE, ResourceAccess::PAGE, ResourceAccess::VIDEO, ResourceAccess::FOLDER, ResourceAccess::LINK]
                );

            $width = $this->getConfig('width', 70) + 10;
            foreach ($resources as $r) {
                $img_path = $r->img_path ?: $this->getConfig('default_path');
                $height = $r->img_height ?: $this->getConfig('img_height');
                if ($img_path) {
                    $img = $this->getDi()->url("data/public/{$img_path}");
                    $this->getDi()->view->headStyle()->appendStyle(<<<CUT
#member-resources li#resource-link-{$r->getAccessType()}-{$r->pk()}-wrapper {
    background: url('{$img}') left top no-repeat;
    padding-left: {$width}px;
    min-height: {$height}px;
}
CUT
                    );
                }
            }
        }
    }
}